﻿using System;
using System.Collections.Generic;
using System.Text;

namespace IndianHealthService.BMXNet.Model
{
    /// <summary>
    /// The visit object is used to determine the current context of the application.  The key wrinkle in the visit
    /// class relates to create a common interface for EHR/VueCentric and WinForm applications.  The EHR/VueCentric supports
    /// visit stubs by default and the WinFramework does not.  If the consumer of the visit object is going to use
    /// it to make changes to RPMS then the visit can not be a stub and must the visit must be created. 
    /// <example>
    /// In either EHR or WinForm applications, always check for a stub before you call RPC's to update RPMS.  If you
    /// do not, the visit may not have been created on RPMS and the visit id will be null.
    /// <code>
    /// if (this.Context.HasVisit)
    /// {             
    ///     if (this.Context.Visit.IsStub &amp;&amp; !this.Context.Visit.Create())
    ///     {
    ///         MessageBox.Show("Unable to establish visit.","Editing Disabled");
    ///         return;
    ///     }
    ///     
    ///     //Call RPC to update RPMS with current visit ien
    /// }
    /// </code>
    /// </example>
    /// </summary>
    public interface Visit
    {
        /// <summary>
        /// This will be non-null if the visit has been created in RPMS, otherwise
        /// it is null and the receiver in considered a stub.
        /// </summary>
        String Ien { get; }

        /// <summary>
        /// Primary provider of the visit
        /// </summary>
        String ProviderName { get; }

        /// <summary>
        /// Hospital Location of the visit
        /// </summary>
        String LocationName { get; }

        /// <summary>
        /// Official  timetamp of the visit
        /// </summary>
        DateTime DateTime { get; }

        /// <summary>
        /// If a visit is transient in the client memory space, it is a stub.
        /// In the EHR visits are often created as stubs (Create New is not checked)
        /// Once data is entered, the visit is created.
        /// The WinForm only supports full created visits therefore IsStub is always false
        /// </summary>
        bool IsStub { get; }
        /// <summary>
        /// Answer true if a future appointment.  
        /// </summary>
        bool IsFutureAppointment { get; }

        /// <summary>
        /// If the receiver is a stub, then call Create() to create a visit in RPMS and
        /// then new data can be added in the context of a created visit.  In the WinForm
        /// edition, this all Visits are created to the method will do nothing yet you should
        /// call it to be consistent.
        /// </summary>
        /// <returns>True if the visit has been created (always True for WinForm)</returns>
        bool Create();
        /// <summary>
        /// In the EHR some visits need user interaction to change from a stub to a created visit.
        /// </summary>
        /// <param name="showGuiToCreateVisit"></param>
        /// <returns></returns>
        bool Create(bool showGuiToCreateVisit);

        /// <summary>
        /// If a visit can not be added to if it is locked. In the WinForm edition, all
        /// visits answered by BMX FIND VISIT are considered unlocked.
        /// </summary>
        bool IsLocked { get;}
    }
}
